<?php
/*--------------------------------------------------------------------
 OnGetSellingUnitPriceEventListener.php 2020-2-28
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Shop\SellingUnit\Database\Price\AttributeInformation\Listener;

use AttributesAjaxHandler;
use Exception;
use Gambio\Shop\ProductModifiers\Modifiers\ValueObjects\AbstractModifierIdentifier;
use Gambio\Shop\SellingUnit\Database\Price\AttributeInformation\Exceptions\NoAttributeOptionValuesIdInModifierCollectionFoundException;
use Gambio\Shop\SellingUnit\Database\Price\AttributeInformation\Service\ReadServiceInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\Interfaces\OnGetSellingUnitPriceEventInterface;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PriceFormatted;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PricePlain;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\PriceStatus;
use GMAttributesCalculator_ORIGIN;
use MainFactory;

require_once dirname(__DIR__, 6) . '/gm/classes/GMAttributesCalculator.php';

/**
 * Class OnGetSellingUnitPriceEventListener
 * @package Gambio\Shop\SellingUnit\Database\Price\AttributeInformation\Listener
 */
class OnGetSellingUnitPriceEventListener
{
    /**
     * @var OnGetSellingUnitPriceEventInterface
     */
    protected $event;
    
    /**
     * @var ReadServiceInterface
     */
    protected $readService;
    
    
    /**
     * OnGetSellingUnitPriceEventListener constructor.
     *
     * @param ReadServiceInterface $readService
     */
    public function __construct(ReadServiceInterface $readService)
    {
        $this->readService = $readService;
    }
    
    
    /**
     * @param OnGetSellingUnitPriceEventInterface $event
     */
    public function __invoke(OnGetSellingUnitPriceEventInterface $event)
    {
        $this->event = $event;
     
        try {
            $calculator = $this->attributesCalculator();
        } catch (NoAttributeOptionValuesIdInModifierCollectionFoundException $exception) {
            unset($exception);
            return;
        }
    
        $formatted = $calculator->calculate($this->event->quantity()->value(), true);
        $plain     = $calculator->calculate($this->event->quantity()->value(), false);
        
        $event->builder()
            ->withPriceFormatted($this->priceFormatted($formatted))
            ->withPricePlain($this->pricePlain($plain))
            ->withStatus($this->priceStatus($event->product()->priceStatus()));
    }
    
    
    /**
     * @return GMAttributesCalculator_ORIGIN
     * @throws NoAttributeOptionValuesIdInModifierCollectionFoundException
     */
    protected function attributesCalculator(): GMAttributesCalculator_ORIGIN
    {
        $optionIdOptionValuesIdDtoCollection = $this->readService->getOptionIdOptionValuesId($this->event->modifiers(),
                                                                                             $this->event->productId());
        MainFactory::load_origin_class('GMAttributesCalculator');
        /** @var GMAttributesCalculator_ORIGIN $result */
        $result = new GMAttributesCalculator_ORIGIN($this->event->productId()->value(),
                                             $optionIdOptionValuesIdDtoCollection->toAssociativeArray(),
                                             $this->event->quantity()->value());
        return $result;
    }
    
    
    /**
     * @param float $price
     *
     * @return PricePlain
     */
    protected function pricePlain(float $price): PricePlain
    {
        return new PricePlain($price);
    }
    
    
    /**
     * @param string $price
     *
     * @return PriceFormatted
     */
    protected function priceFormatted(string $price): PriceFormatted
    {
        return new PriceFormatted($price);
    }

    /**
     * @param int $priceStatus
     * @return PriceStatus
     */
    protected function priceStatus(int $priceStatus) : PriceStatus
    {
        return new PriceStatus($priceStatus);

    }
}